'use strict';

/* --------------------------------------------------------------
 product_image_list.js 2020-01-23
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Product Images List Controller
 *
 * This controller handles the images sort
 *
 * @module Controllers/product_image_list
 */

gx.controllers.module(
// ------------------------------------------------------------------------
// CONTROLLER NAME
// ------------------------------------------------------------------------
'product_image_list',

// ------------------------------------------------------------------------
// CONTROLLER LIBRARIES
// ------------------------------------------------------------------------
[jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.js', 'loading_spinner'],

// ------------------------------------------------------------------------
// CONTROLLER BUSINESS LOGIC
// ------------------------------------------------------------------------
function (data) {
  'use strict';

  var _this = this;

  var module = {};

  /**
   * Edit modal element
   *
   * @type {jQuery|HTMLElement}
   */
  var $editModal = $('.edit-panel');

  /**
   * Edit collection modal element
   *
   * @type {jQuery|HTMLElement}
   */
  var $editCollectionModal = $('.edit-collection-panel');

  /**
   * Confirm modal element
   *
   * @type {jQuery|HTMLElement}
   */
  var $confirmModal = $('.confirm-modal-panel');

  /**
   * Toast message element
   *
   * @type {jQuery|HTMLElement}
   */
  var $toastMessage = $('.request-status-wrapper');

  /**
   * Selected image collection value
   */
  var initialSelectedList = data.selectedList;

  /**
   *
   * @type {*}
   */
  var appUrl = jse.core.config.get('appUrl');

  /**
   * Load spinner instance
   */
  var loadingSpinner = null;

  /**
   *
   */
  var toastTimeoutID = null;

  /**
   * Default options
   *
   * @type {{sortable: {cursor: string, containment: string, sortableList: string, handle: string, placeholder: string, axis: string, opacity: number, items: string}, filemanager: {subFolder: string, top: number, left: number, width: number, id: string, lang: string, height: number}}}
   */
  var defaults = {
    sortable: {
      items: 'li.row',
      axis: 'y',
      cursor: 'move',
      handle: '.sort-handle',
      containment: 'document',
      opacity: 0.75,
      placeholder: 'col-md-12 list-element sort-placeholder',
      sortableList: 'ul.configuration-image-list'
    },
    filemanager: {
      id: 'add-image-collection',
      subFolder: 'images/product_images/original_images',
      popup: 1,
      lang: 'de',
      width: 800,
      height: 600,
      top: 100,
      left: 100,
      useFileManager: false
    }

    /**
     * Sortable options
     */
  };var sortableOptions = $.extend(true, {}, defaults.sortable, data);

  /**
   * File manager options
   */
  var filemanagerOptions = $.extend(true, {}, defaults.filemanager, data);

  var fileManager = null;

  /**
   * Sortable list element.
   *
   * @type {jQuery}
   */
  var $sortableList = $(sortableOptions.sortableList);

  /**
   * Handler when updating the order of the image list
   */
  var handleOnSortUpdate = async function handleOnSortUpdate() {
    loading(true);

    try {
      var newOrder = $sortableList.sortable('toArray', { attribute: 'data-list-element-id' }).reduce(function (newValue, currentValue, currentIndex) {
        newValue.push({
          imageId: parseInt(currentValue),
          sortIndex: currentIndex
        });
        return newValue;
      }, []);

      var result = await updateImagesSort({ sort: JSON.stringify(newOrder) });

      if (result.success) {
        toastSuccess(result.message);
      } else {
        toastError(result.message);
      }
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Event listener for add image button
   */
  var addImageButtonEventListener = function addImageButtonEventListener() {
    $('.add-image').click(function () {
      var id = filemanagerOptions.id,
          subFolder = filemanagerOptions.subFolder,
          lang = filemanagerOptions.lang,
          width = filemanagerOptions.width,
          height = filemanagerOptions.height,
          top = filemanagerOptions.top,
          left = filemanagerOptions.left,
          popup = filemanagerOptions.popup,
          useFileManager = filemanagerOptions.useFileManager;


      if (!useFileManager) {
        toastError(jse.core.lang.translate('RESPONSIVE_FILEMANAGER_REQUIRED', 'product_image_lists'));
        return;
      }

      var urlParams = ['field_id=' + id, 'crossdomain=1', 'sub_folder=' + subFolder, 'lang=' + lang, 'popup=' + popup].join('&');

      fileManager = parent.window.open(appUrl + '/ResponsiveFilemanager/filemanager/dialog.php?' + urlParams, 'ResponsiveFilemanager', 'scrollbars=1,width=' + width + ',height=' + height + ',top=' + top + ',left=' + left);
    });
  };

  var addResponsiveFileManagerEventListener = function addResponsiveFileManagerEventListener() {
    var onMessage = async function onMessage(event) {
      var data = event.data,
          sender = data.sender,
          url = data.url,
          field_id = data.field_id;


      if (sender !== 'responsivefilemanager' || field_id !== filemanagerOptions.id) {
        return;
      }

      var correctUrl = url.replace(/([^:]\/)\/+/g, '$1');

      await handleAddImageToCollection({
        url: correctUrl,
        localPath: normalizeLocalPath(correctUrl)
      });

      fileManager.close();
      fileManager = null;
    };

    window.addEventListener('message', onMessage, false);
  };

  /**
   * Removes the origin domain from the image path
   *
   * @param url
   * @returns {*}
   */
  var normalizeLocalPath = function normalizeLocalPath(url) {
    var regex = appUrl + '(/)?';

    return url.replace(new RegExp(regex, 'g'), '');
  };

  /**
   * Event listener for creating a new image collection
   */
  var createNewCollectionButtonEventListener = function createNewCollectionButtonEventListener() {
    $('#create-new-collection').on('click', function () {
      clearCreateNewCollectionInput();

      $('.selected-collection-wrapper').addClass('create-new-collection');
      $('.select-collection').hide();
      $('.create-collection').show();
      $('.create-collection input').focus();

      if (hasCollection()) {
        $('#select-collection').show();
      }

      $(this).hide();
    });
  };

  /**
   * Event listener for clicking on "Select collection" text/button
   */
  var selectCollectionButtonEventListener = function selectCollectionButtonEventListener() {
    $('#select-collection').on('click', function () {
      $('.selected-collection-wrapper').removeClass('create-new-collection');
      $('.create-collection').hide();
      $('.select-collection').show();
      $('#create-new-collection').show();
      $(this).hide();
    });
  };

  /**
   * Event listener for opening the Edit image modal
   */
  var openEditImageModalEventListener = function openEditImageModalEventListener() {
    $(document).on('click', '.edit-image', function () {
      var $parent = $(this).parents('.collection-image-wrapper'),
          data = $parent.data('image');

      handleEditImageModal({
        id: data.id,
        src: data.webFilePath,
        localPath: normalizeLocalPath(data.webFilePath),
        titles: data.titles,
        altTitles: data.altTitles
      });
    });
  };

  /**
   * Event listener for closing Edit image modal
   */
  var closeEditModalButtonEventListener = function closeEditModalButtonEventListener() {
    $('.edit-panel .edit-modal .close-btn').on('click', handleCloseEditImageModal);
  };

  /**
   * Event listener for canceling/closing the Edit image modal
   */
  var cancelEditModalEventListener = function cancelEditModalEventListener() {
    $(document).on('click', '.edit-modal-cancel', handleCloseEditImageModal);
  };

  /**
   * Event listener for hitting the save button inside Edit Image Modal
   */
  var saveEditModalEventListener = function saveEditModalEventListener() {
    $editModal.on('click', '.edit-modal-save', handleSaveImageModal);
  };

  /**
   * Event listener for closing Edit collection modal
   */
  var closeEditCollectionModalButtonEventListener = function closeEditCollectionModalButtonEventListener() {
    $('.edit-collection-panel .edit-modal .close-btn').on('click', handleCloseEditCollectionModal);
  };

  /**
   * Event listener for opening the Edit collection modal
   */
  var openEditCollectionModalEventListener = function openEditCollectionModalEventListener() {
    $(document).on('click', '.edit-collection', handleEditCollectionModal);
  };

  /**
   * Event listener for canceling/closing the Edit image modal
   */
  var cancelEditCollectionModalButtonEventListener = function cancelEditCollectionModalButtonEventListener() {
    $(document).on('click', '.edit-collection-modal-cancel', handleCloseEditCollectionModal);
  };

  /**
   * Listener for clicking on save button inside Edit Collection Modal
   */
  var updateEditCollectionModalButtonEventListener = function updateEditCollectionModalButtonEventListener() {
    $editCollectionModal.on('click', '.edit-collection-modal-save', handleUpdateCollectionModal);
  };

  /**
   * Handler for closing Edit Image Modal
   */
  var handleCloseEditCollectionModal = function handleCloseEditCollectionModal() {
    $editCollectionModal.fadeOut(function () {
      return clearModalInputs($editCollectionModal);
    });
  };

  /**
   * Event listener for deleting image button
   */
  var deleteImageButtonEventListener = function deleteImageButtonEventListener() {
    $(document).on('click', '.delete-image', function () {
      handleDeleteImage($(this).parents('.collection-image-wrapper'));
    });
  };

  /**
   * Event listener when changing the image collection dropdown
   */
  var imageCollectionOnChangeEventListener = function imageCollectionOnChangeEventListener() {
    $('#combi_image_collection').change(function () {
      var val = parseInt($(this).val());
      handleOnChangeImageCollection(val);
    });
  };

  /**
   * Event listener for creating new collection button
   */
  var createCollectionButtonEventListener = function createCollectionButtonEventListener() {
    $('button.create-new-collection').on('click', async function () {
      var $input = $('input[name=new-collection-name]'),
          name = $input.val();

      if (!name) {
        $input.addClass('error');

        return false;
      }

      await handleCreateNewCollection(name);
    });
  };

  /**
   * Event listener for deleting collection list button
   */
  var deleteCollectionButtonEventListener = function deleteCollectionButtonEventListener() {
    $('button.delete-collection').on('click', async function () {
      handleDeleteCollection($('#combi_image_collection option:selected').val());
    });
  };

  /**
   * Prevents submit the modal forms by pressing enter.
   * It triggers the save button instead of submitting it.
   *
   */
  var preventSubmitFormModals = function preventSubmitFormModals() {
    $editModal.find('#edit-image-form').submit(function (e) {
      $editModal.find('.edit-modal-save').trigger('click');
      return false;
    });
    $editCollectionModal.find('#edit-collection-form').submit(function (e) {
      $editCollectionModal.find('.edit-collection-modal-save').trigger('click');
      return false;
    });
  };

  /**
   * Handler for closing Edit Image Modal
   */
  var handleCloseEditImageModal = function handleCloseEditImageModal() {
    if ($editModal.find('input[name=id]').val() === '0') {
      removeLastImageElement();
    }

    $editModal.fadeOut(function () {
      return clearModalInputs($editModal);
    });
  };

  /**
   * Handler to add an image to a collection
   *
   * @returns {void}
   */
  var handleSaveImageModal = async function handleSaveImageModal() {
    if (!isValidEditModalData($editModal)) {
      return;
    }

    loading(true);

    try {
      var formData = parseEditModalData($('form#edit-image-form').serializeArray()),
          result = null;

      if (formData.id !== '0') {
        formData.imageId = formData.id;
        delete formData.id;
        result = await updateImageTexts(formData);
      } else {
        delete formData.id;
        result = await addImageToCollection(formData);
      }

      resetImagesWrapper();
      loadImagesCollection(getSelectedList().id);
      $editModal.fadeOut(function () {
        return clearModalInputs($editModal);
      });

      var _result = result,
          success = _result.success,
          _result$message = _result.message,
          message = _result$message === undefined ? null : _result$message;


      if (success) {
        toastSuccess(message || jse.core.lang.translate('MODAL_SUCCESS_TEXT', 'product_image_lists'));
      } else {
        toastError(message || jse.core.lang.translate('MODAL_ERROR_TEXT', 'product_image_lists'));
      }
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Handler to update the collection's name
   *
   * @returns {void}
   */
  var handleUpdateCollectionModal = async function handleUpdateCollectionModal() {
    if (!isValidEditModalData($editCollectionModal)) {
      return;
    }

    loading(true);
    try {
      var formData = parseEditModalData($('form#edit-collection-form').serializeArray());

      var id = formData.id,
          name = formData.name;

      var _ref = await updateCollectionName({ listId: id, listName: name }),
          success = _ref.success,
          message = _ref.message;

      if (success) {
        toastSuccess(message);
      } else {
        toastError(message);
      }

      loadCollections(id);
      $editCollectionModal.fadeOut();
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Validates edit modals
   *
   * @param $modal
   * @returns {boolean}
   */
  var isValidEditModalData = function isValidEditModalData($modal) {
    $modal.find('form input').removeClass('error');

    var $form = $modal.find('form');
    var errors = 0;

    $form.find('[class*=required]').each(function (index, element) {
      if ($(element).val() === '') {
        $(element).addClass('error');
        errors++;
      }
    });

    return errors === 0;
  };

  /**
   * Clears the inputs for the given modal element
   *
   * @param $modal
   */
  var clearModalInputs = function clearModalInputs($modal) {
    $modal.find('input').each(function (index, element) {
      $(element).val('').removeClass('error');
    });
  };

  /**
   * Parse edit modal form data to send to back end
   *
   * @param formData
   * @returns {{*}}
   */
  var parseEditModalData = function parseEditModalData(formData) {
    var parsedData = { titles: [], altTitles: [] };

    formData.forEach(function (element) {
      if (/\[\w+\]/gi.test(element.name)) {
        var key = element.name.replace(/.*\[(\w+)\]/, '$1'),
            value = element.name.replace(/(.*)\[\w+\]/, '$1');

        parsedData[value].push({
          value: element.value,
          languageCode: key
        });
      } else {
        parsedData[element.name] = element.value;
      }
    });

    parsedData.titles = JSON.stringify(parsedData.titles);
    parsedData.altTitles = JSON.stringify(parsedData.altTitles);

    return parsedData;
  };

  var openConfirmModal = function openConfirmModal() {
    return new Promise(function (resolve) {
      $confirmModal.show();

      $confirmModal.on('click', '.confirm-modal-cancel, .confirm-modal .close-btn', function () {
        resolve(false);
        $confirmModal.hide();
      });
      $confirmModal.find('.confirm-modal-confirm').click(function () {
        resolve(true);
        $confirmModal.hide();
      });
    });
  };

  /**
   * Handler when deleting a image from the list
   * @param $imageWrapper
   */
  var handleDeleteImage = async function handleDeleteImage($imageWrapper) {
    var canProceed = await openConfirmModal();

    if (!canProceed) {
      return;
    }

    loading(true);

    try {
      var imageId = parseInt($imageWrapper.data('image').id);
      var message = jse.core.lang.translate('MODAL_SUCCESS_TEXT', 'product_image_lists');

      if (imageId) {
        var result = await deleteImageById(imageId);
        message = result.message;
      }

      $imageWrapper.fadeOut(function () {
        $(this).remove();

        if (!hasImage()) {
          setToNoImagesDisplay();
        }
      });

      toastSuccess(message);
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Clears the new collection input text
   *
   * @returns {void}
   */
  var clearCreateNewCollectionInput = function clearCreateNewCollectionInput() {
    $('input[name=new-collection-name]').removeClass('error').val('');
  };

  /**
   * Appends an image to the image wrapper
   *
   * @param $imageBlock
   */
  var appendImageToCollection = function appendImageToCollection($imageBlock) {
    $('.selected-collection-wrapper').removeClass('create-new-collection no-image-selected');
    $('.collection-images-wrapper').css('display', 'flex').append($imageBlock);
  };

  /**
   * Resets/clears the image wrapper
   *
   * @returns {jQuery}
   */
  var resetImagesWrapper = function resetImagesWrapper() {
    return $('.collection-images-wrapper').html('');
  };

  /**
   * Replaces the collection dropdown with new collection list
   *
   * @param collections
   * @param selected
   */
  var replaceCollections = function replaceCollections(collections) {
    var selected = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;

    var $collection = $('select#combi_image_collection');
    $collection.html('').append($('<option/>').val(0).html(''));

    collections.forEach(function (element) {
      var $option = $('<option/>').val(element.id).html(element.name);

      if (selected !== null && parseInt(selected) === element.id) {
        $option.prop('selected', true);
      }

      $collection.append($option);
    });
  };

  /**
   * Initialize the collections list
   *
   * @returns {void}
   */
  var loadCollections = async function loadCollections() {
    var selected = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : null;

    var collections = await getCollection();

    replaceCollections(collections, selected);

    if (!collections.length) {
      setToNoImagesDisplay();
      setToNoCollectionsDisplay();
      return;
    }

    enableImageSelection();
  };

  /**
   * Initialize the images collection by a given collection ID
   *
   * @param collection_id
   * @returns {void}
   */
  var loadImagesCollection = async function loadImagesCollection(collection_id) {
    var images = await getImageCollection(collection_id);

    if (!images.length) {
      setToNoImagesDisplay();
      return;
    }
    // $('.selected-collection-wrapper').removeClass('disabled-image-selection')

    images.sort(function (a, b) {
      return a.sortOrder - b.sortOrder;
    }).forEach(function (element) {
      var $imageBlock = $(imageBlockTemplate({ id: element.id, src: element.webFilePath }));

      $imageBlock.data('image', element);
      appendImageToCollection($imageBlock);
    });
  };

  /**
   * Hides the collection image wrapper element
   */
  var setToNoImagesDisplay = function setToNoImagesDisplay() {
    $('.selected-collection-wrapper').addClass('no-image-selected');
    $('div.collection-images-wrapper').hide();
  };

  /**
   * Hide elements when we don't have collections
   */
  var setToNoCollectionsDisplay = function setToNoCollectionsDisplay() {
    $('button#select-collection').hide();
    $('button#create-new-collection').trigger('click');
  };

  /**
   * Disables the image placeholder and the edit/delete buttons of the image collection
   */
  var disableImageSelection = function disableImageSelection() {
    $('.selected-collection-wrapper').addClass('disabled-image-selection');
    $('button.edit-collection, button.delete-collection').attr('disabled', true);
  };

  /**
   * Enables the image placeholder and the edit/delete buttons of the image collection
   */
  var enableImageSelection = function enableImageSelection() {
    $('.selected-collection-wrapper').removeClass('disabled-image-selection');
    $('button.edit-collection, button.delete-collection').removeAttr('disabled');
  };

  /**
   * Returns true if there is at least one collection
   * We check if it is bigger than 1 because the first element it's a blank option
   *
   * @returns {boolean}
   */
  var hasCollection = function hasCollection() {
    return $('#combi_image_collection option').length > 1;
  };

  /**
   * Checks if the images wrapper has images
   *
   * @returns {boolean}
   */
  var hasImage = function hasImage() {
    return $('.collection-images-wrapper > .collection-image-wrapper').length > 0;
  };

  /**
   * Removes the last image element from the images container
   */
  var removeLastImageElement = function removeLastImageElement() {
    // remove last image
    var $lastImageElement = $('.collection-images-wrapper > .collection-image-wrapper:last-child');
    $lastImageElement.remove();

    if (!hasImage()) {
      setToNoImagesDisplay();
    }
  };

  /**
   * Request to get all the image collections
   *
   * @returns {Promise}
   */
  var getCollection = function getCollection() {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListReadAjax'].join('');

    return $.get(url, function (response) {
      return response;
    });
  };

  /**
   * Get all images from a given collection
   *
   * @param collection_id
   * @returns {Promise}
   */
  var getImageCollection = function getImageCollection(collection_id) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListReadAjax/list', '&id=' + collection_id].join('');

    return $.get(url).then(function (response) {
      return response[0].images || [];
    });
  };

  /**
   * Request to create a new collection list
   *
   * @param listName
   */
  var createCollection = function createCollection(listName) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListCreateAjax/imageList'].join('');

    return $.post(url, { listName: listName }, function (response) {
      return response;
    });
  };

  /**
   * Request to delete a collection list
   *
   * @param id
   * @param modifierId
   * @param modifierType
   * @returns {Promise}
   */
  var deleteCollection = function deleteCollection(_ref2) {
    var id = _ref2.id,
        modifierId = _ref2.modifierId,
        modifierType = _ref2.modifierType;

    var url = [appUrl, '/admin/admin.php?do=ProductImageListDeleteAjax/deleteImageListById', '&id=' + id + '&modifierId=' + modifierId + '&modifierType=' + modifierType].join('');

    return $.ajax({
      url: url,
      type: 'DELETE'
    }).then(function (response) {
      return response;
    });
  };

  /**
   * Request to add an image to a collection list
   *
   * @param parsedData
   * @returns {*}
   */
  var addImageToCollection = function addImageToCollection(parsedData) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListCreateAjax/image'].join('');

    return $.post(url, parsedData, function (response) {
      return response;
    }, 'json');
  };

  /**
   * Updates the text and alt text for the image
   *
   * @param parsedData
   * @returns {*}
   */
  var updateImageTexts = function updateImageTexts(parsedData) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListUpdateAjax/updateImageText'].join('');

    return $.post(url, parsedData, function (response) {
      return response;
    }, 'json');
  };

  /**
   * Request to update a collection name
   *
   * @param parsedData
   * @returns {*}
   */
  var updateCollectionName = function updateCollectionName(parsedData) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListUpdateAjax/updateImageListName'].join('');

    return $.post(url, parsedData, function (response) {
      return response;
    }, 'json');
  };

  /**
   * Deletes an image from the current image collection
   *
   * @param id
   * @returns {Promise}
   */
  var deleteImageById = function deleteImageById(id) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListDeleteAjax/deleteImageById&id=' + id].join('');

    return $.ajax({
      url: url,
      type: 'DELETE'
    }).then(function (response) {
      return response;
    });
  };

  /**
   * Updates the order of the images
   *
   * @param parsedData
   * @returns {*}
   */
  var updateImagesSort = function updateImagesSort(parsedData) {
    var url = [appUrl, '/admin/admin.php?do=ProductImageListUpdateAjax/updateImagesSort'].join('');

    return $.post(url, parsedData, function (response) {
      return response;
    }, 'json');
  };

  /**
   * Handler to create a new collection button event
   *
   * @param name
   * @returns {Promise<void>}
   */
  var handleCreateNewCollection = async function handleCreateNewCollection(name) {
    loading(true);
    try {
      // Ajax request to create a new
      var result = await createCollection(name);

      // Get image collection list
      resetImagesWrapper();
      await loadCollections();
      await loadImagesCollection($('select#combi_image_collection option:last-child').prop('selected', true).val());

      // Trigger "Select collection" button
      $('#select-collection').trigger('click');

      if (result.success) {
        toastSuccess(result.message);
      } else {
        toastError(result.message);
      }
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Handler for deleting collection button
   *
   * @param id
   * @returns {void}
   */
  var handleDeleteCollection = async function handleDeleteCollection(id) {
    var canProceed = await openConfirmModal();

    if (!canProceed) {
      return;
    }

    loading(true);

    try {
      var modifierId = $('input:hidden[name=modifierId]').val(),
          modifierType = $('input:hidden[name=modifierType]').val();

      var params = {
        id: id,
        modifierId: modifierId,
        modifierType: modifierType
      };
      var deleteCollectionResult = await deleteCollection(params);

      if (!deleteCollectionResult.success) {
        loading(false);
        toastError(deleteCollectionResult.message);
        return;
      }

      await loadCollections();

      var $firstOption = $('select#combi_image_collection option:first-child'),
          firstOptionVal = parseInt($firstOption.val());

      resetImagesWrapper();

      if (!firstOptionVal) {
        disableImageSelection();
      } else {
        await loadImagesCollection($firstOption.val());
      }

      toastSuccess(deleteCollectionResult.message);
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  /**
   * Handler for saving an image to a collection
   *
   * @param path
   * @returns {void}
   */
  var handleAddImageToCollection = async function handleAddImageToCollection(_ref3) {
    var url = _ref3.url,
        localPath = _ref3.localPath;

    var params = { id: 0, src: url, localPath: localPath };
    var $image = $(imageBlockTemplate(params));

    appendImageToCollection($image);
    handleEditImageModal(params);
  };

  /**
   * Handler for open the Edit image modal button
   *
   * @param id
   * @param src
   * @param localPath
   * @param titles
   * @param altTitles
   */
  var handleEditImageModal = function handleEditImageModal(_ref4) {
    var id = _ref4.id,
        src = _ref4.src,
        localPath = _ref4.localPath,
        _ref4$titles = _ref4.titles,
        titles = _ref4$titles === undefined ? [] : _ref4$titles,
        _ref4$altTitles = _ref4.altTitles,
        altTitles = _ref4$altTitles === undefined ? [] : _ref4$altTitles;

    $editModal.find('#collection-image-src').attr('src', src);
    $editModal.find('input[name=id]').val(id);
    $editModal.find('input[name=localPath]').val(localPath);
    $editModal.find('input[name=listId]').val(getSelectedList().id);

    titles.forEach(function (element) {
      $('#image-title-' + element.languageCode.toLowerCase()).val(element.value);
    });

    altTitles.forEach(function (element) {
      $('#image-alt-' + element.languageCode.toLowerCase()).val(element.value);
    });

    $editModal.fadeIn(function () {
      return focusFirstInputText($editModal);
    });
  };

  /**
   * Handler for open the Edit image modal button
   */
  var handleEditCollectionModal = function handleEditCollectionModal() {
    $editCollectionModal.find('input[name=name]').val(getSelectedList().name);
    $editCollectionModal.find('input[name=id]').val(getSelectedList().id);
    // Ajax request
    $editCollectionModal.fadeIn(function () {
      return focusFirstInputText($editCollectionModal);
    });
  };

  /**
   * Handles image list on change event
   *
   * @param val
   * @returns {Promise<void>}
   */
  var handleOnChangeImageCollection = async function handleOnChangeImageCollection(val) {
    loading(true);

    try {
      if (!val) {
        disableImageSelection();
        return;
      }

      enableImageSelection();
      resetImagesWrapper();
      await loadImagesCollection(val);
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
    }
  };

  var focusFirstInputText = function focusFirstInputText($element) {
    $element.find('input[type=text]:first').focus();
  };

  /**
   * Creates the single image wrapper
   *
   * @param data
   * @returns {string}
   */
  var imageBlockTemplate = function imageBlockTemplate(data) {
    var _data$src = data.src,
        src = _data$src === undefined ? '//placehold.it/100x100' : _data$src,
        id = data.id,
        mainImageText = jse.core.lang.translate('MAIN_IMAGE_LABEL', 'product_image_lists');


    return '<div class="collection-image-wrapper" data-list-element-id="' + id + '">\n' + '    <div class="actions">\n' + '        <a class="btn btn-primary move-image sort-handle" href="javascript:;">\n' + '            <i class="fa fa-arrows"></i>\n' + '        </a>\n' + '        <a class="btn btn-primary edit-image" href="javascript:;">\n' + '            <i class="fa fa-pencil"></i>\n' + '        </a>\n' + '        <a class="btn btn-danger delete-image" href="javascript:;">\n' + '            <i class="fa fa-trash"></i>\n' + '        </a>\n' + '    </div>\n' + ('    <span class="main-image">' + mainImageText + '</span>\n') + ('    <img alt="alt img" class="collection-image" src="' + src + '">\n') + '</div>';
  };

  /**
   * Gets the selected collection list
   *
   * @returns {{name: string, id: string}}
   */
  var getSelectedList = function getSelectedList() {
    var $selectedOption = $('#combi_image_collection option:selected');
    return {
      id: $selectedOption.val(),
      name: $selectedOption.html()
    };
  };

  /**
   * Bind events
   */
  var addEventListeners = function addEventListeners() {
    // Add event listeners
    addResponsiveFileManagerEventListener();

    createNewCollectionButtonEventListener();
    selectCollectionButtonEventListener();
    closeEditModalButtonEventListener();
    openEditImageModalEventListener();
    deleteImageButtonEventListener();
    imageCollectionOnChangeEventListener();
    addImageButtonEventListener();

    createCollectionButtonEventListener();
    deleteCollectionButtonEventListener();

    // Modal events
    preventSubmitFormModals();
    saveEditModalEventListener();
    cancelEditModalEventListener();

    closeEditCollectionModalButtonEventListener();
    openEditCollectionModalEventListener();
    cancelEditCollectionModalButtonEventListener();
    updateEditCollectionModalButtonEventListener();
  };

  /**
   * Toast "plugin"
   *
   * @param message
   * @param type
   */
  var toast = function toast(message, type) {
    var className = 'status-' + type;

    $toastMessage.html(message).removeClass(function (index, className) {
      return (className.match(/(^|\s)status-\S+/g) || []).join(' ');
    }).addClass(className).stop().fadeIn();

    clearTimeout(toastTimeoutID);

    toastTimeoutID = setTimeout(function () {
      $toastMessage.fadeOut(function () {
        $(_this).removeClass(className);
      });
    }, 3000);
  };

  /**
   * Shows success toast
   *
   * @param message
   */
  var toastSuccess = function toastSuccess(message) {
    return toast(message, 'success');
  };

  /**
   * Shows error toast
   *
   * @param message
   */
  var toastError = function toastError(message) {
    return toast(message, 'error');
  };

  /**
   * Handles loading spinner
   *
   * @param isLoading
   */
  var loading = function loading(isLoading) {
    if (isLoading) {
      loadingSpinner = jse.libs.loading_spinner.show($('#product-image-list-wrapper'), 9999);
      return;
    }

    jse.libs.loading_spinner.hide(loadingSpinner);
    loadingSpinner = null;
  };

  /**
   * Initialize the module
   *
   * @param done
   * @returns {void}
   */
  module.init = async function (done) {
    addEventListeners();

    $sortableList.sortable(sortableOptions).on('sortupdate', handleOnSortUpdate).disableSelection();

    loading(true);

    try {
      // Get image collections list
      await loadCollections(initialSelectedList);

      // Get images related with the current collection
      var selectedValue = initialSelectedList ? initialSelectedList : $('select#combi_image_collection option:selected').val();

      selectedValue = parseInt(selectedValue);

      if (!isNaN(selectedValue) && selectedValue > 0) {
        await loadImagesCollection(selectedValue);
      } else {
        disableImageSelection();
      }
    } catch (e) {
      toastError(e.message);
    } finally {
      loading(false);
      done();
    }
  };

  return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
